<?php

/**
 * Project: RBI Reeport
 * Author: Grant McClure <grant.mcclure@t4bi.com>
 * File: rbi_report.lib.php
 * Version: 1.0
 */

require(RBI_REPORT_DIR . 'libs/Wkhtmltopdf.php');

/**
 * RBI Report
 *
 */
class RBI_Report {

  // database object
  var $pdo = null;
  // smarty template object
  var $tpl = null;
  // error messages
  var $error = null;

  /* set database settings here! */
  var $dbtype = 'mysql';
  var $dbname = 'aggdata_dev';
  var $dbhost = 'localhost';
  var $dbuser = 'reports';
  var $dbpass = '';


  /**
  * class constructor
  */
  function __construct() {

    // instantiate the pdo object
    try {
      $dsn = "{$this->dbtype}:host={$this->dbhost};dbname={$this->dbname}";
      $this->pdo = new PDO($dsn,$this->dbuser,$this->dbpass);
    } catch (PDOException $e) {
      print "Error!: " . $e->getMessage();
      die();
    }   

    // instantiate the template object
    $this->tpl = new RBI_Reports_Smarty;

  }

  function displayReportSelector() {
    // assign error message
    $this->tpl->assign('error', $this->error);
    $allArray = $this->getLocationValues(1);
    $regionArray = $this->getLocationValues(2);
    $assocArray = $this->getLocationValues(3);
    $countyArray = $this->getLocationValues(4);
    $zipArray = $this->getLocationValues(5);
    $altRegionArray = $this->getLocationValues(6);

    $monthArrayTemp = $this->getReportMonths();
    $monthArray = array();
    foreach ($monthArrayTemp as $row) {
     $row["timestamp"] = mktime(0, 0, 0, substr($row["time_period"],4,2), 1, substr($row["time_period"],0,4));
     $monthArray[] = $row;
    }

    $yearArrayTemp = $this->getReportYears();
    foreach ($yearArrayTemp as $row) {
     $row["timestamp"] = mktime(0, 0, 0, 1, 1, substr($row["time_period"],0,4));
     $yearArray[] = $row;
    }

    $this->tpl->assign('allArray', $allArray);
    $this->tpl->assign('regionArray', $regionArray);
    $this->tpl->assign('assocArray', $assocArray);
    $this->tpl->assign('countyArray', $countyArray);
    $this->tpl->assign('zipArray', $zipArray);
    $this->tpl->assign('altRegionArray', $altRegionArray);
    $this->tpl->assign('monthArray', $monthArray);
    $this->tpl->assign('yearArray', $yearArray);

    $this->tpl->display("report_selector.tpl");
  }



  function createReport($agg_type, $loc_type, $loc_id, $time_period) {
    // assign error message
    $this->tpl->assign('error', $this->error);

    $time_period_last_year = $this->getPreviousYearTimePeriod($time_period);

    $monthForTimestamp = substr($time_period,4,2);
    if (!$monthForTimestamp) {
      $monthForTimestamp=1; // default to January for annual report
    }
    $this->tpl->assign('thisPeriodTimestamp', mktime(0, 0, 0, $monthForTimestamp, 1, substr($time_period,0,4)));
    $this->tpl->assign('lastPeriodTimestamp', mktime(0, 0, 0, $monthForTimestamp, 1, substr($time_period_last_year,0,4)));

    $locData = $this->getLocationData($loc_type, $loc_id);
    $this->tpl->assign('locationName', $locData['l_desc']);
    
    // Get Core Data (All Facet)
    $coreDataThisYear = $this->getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period);
    $this->tpl->assign('coreDataThisYear', $coreDataThisYear);
    $coreDataLastYear = $this->getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period_last_year);
    $this->tpl->assign('coreDataLastYear', $coreDataLastYear);
    $this->tpl->assign('coreDataDiffYear', $this->getDataDiff($coreDataThisYear, $coreDataLastYear));

    // Get Attached Data
    $attachedDataThisYear = $this->getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period, 2, 1003);
    $this->tpl->assign('attachedDataThisYear', $attachedDataThisYear);
    $attachedDataLastYear = $this->getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period_last_year, 2, 1003);
    $this->tpl->assign('attachedDataLastYear', $attachedDataLastYear);
    $this->tpl->assign('attachedDataDiffYear', $this->getDataDiff($attachedDataThisYear, $attachedDataLastYear));

    // Get Detached Data
    $detachedDataThisYear = $this->getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period, 2, 1001);
    $this->tpl->assign('detachedDataThisYear', $detachedDataThisYear);
    $detachedDataLastYear = $this->getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period_last_year, 2, 1001);
    $this->tpl->assign('detachedDataLastYear', $detachedDataLastYear);
    $this->tpl->assign('detachedDataDiffYear', $this->getDataDiff($detachedDataThisYear, $detachedDataLastYear));

    // Get Financing Data
    $financingDataThisYear = $this->getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period, 11);
    $this->tpl->assign('financingDataThisYear', $financingDataThisYear);

    // Get DOM Data
    $domDataThisYear = $this->getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period, 5);
    $this->tpl->assign('domDataThisYear', $domDataThisYear);

    // Get Sold Price, Home/Type Ownership, Bedrooms Detail Data
    $brDetailDataThisYear = $this->getDetailAggregationData($agg_type, $loc_type, $loc_id, $time_period, 4, 2, 12);
    $this->tpl->assign('brDetailDataThisYear', $brDetailDataThisYear);
    //$brDetailDataLastYear = $this->getDetailAggregationData($agg_type, $loc_type, $loc_id, $time_period_last_year, 4, 2, 12);
    //$this->tpl->assign('brDetailDataLastYear', $brDetailDataLastYear);
    //$this->tpl->assign('brDetailDataDiffYear', $this->getDataDiff($brDetailDataThisYear, $brDetailDataLastYear));

    // Get Total, Home/Type Ownership, Bedrooms Detail Data
    $tbrDetailDataThisYear = $this->getDetailAggregationData($agg_type, $loc_type, $loc_id, $time_period, 1, 2, 12);
    $this->tpl->assign('tbrDetailDataThisYear', $tbrDetailDataThisYear);
    $tbrDetailDataLastYear = $this->getDetailAggregationData($agg_type, $loc_type, $loc_id, $time_period_last_year, 1, 2, 12);
    $this->tpl->assign('tbrDetailDataLastYear', $tbrDetailDataLastYear);
    $this->tpl->assign('tbrDetailDataDiffYear', $this->getDataDiff($tbrDetailDataThisYear, $tbrDetailDataLastYear));

    // Get Sold Price, Home/Type Ownership Detail Data 
    $allSoldDetailDataThisYear = $this->getDetailAggregationData($agg_type, $loc_type, $loc_id, $time_period, 4, 2, 1);
    $this->tpl->assign('allSoldDetailDataThisYear', $allSoldDetailDataThisYear);

    // Get List Price, Home/Type Ownership Detail Data
    $allListDetailDataThisYear = $this->getDetailAggregationData($agg_type, $loc_type, $loc_id, $time_period, 3, 2, 1);
    $this->tpl->assign('allListDetailDataThisYear', $allListDetailDataThisYear);
    //$allDetailDataLastYear = $this->getDetailAggregationData($agg_type, $loc_type, $loc_id, $time_period_last_year, 3, 2, 1);
    //$this->tpl->assign('allDetailDataLastYear', $allDetailDataLastYear);
    //$this->tpl->assign('allDetailDataDiffYear', $this->getDataDiff($allDetailDataThisYear, $allDetailDataLastYear));

    // Get Total, Home/Type Ownership Detail Data 
    $tallCoreDataThisYear = $this->getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period, 2);
    $this->tpl->assign('tallCoreDataThisYear', $tallCoreDataThisYear);
    $tallCoreDataLastYear = $this->getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period_last_year, 2);
    $this->tpl->assign('tallCoreDataLastYear', $tallCoreDataLastYear);
    $this->tpl->assign('tallCoreDataDiffYear', $this->getDataDiff($tallCoreDataThisYear, $tallCoreDataLastYear));

    // Add Facet Lookups
    $spFacetList = $this->getFacetValues(4);
    $this->tpl->assign('spFacetList', $spFacetList);
    $brFacetList = $this->getFacetValues(7);
    $this->tpl->assign('brFacetList', $brFacetList);
    $htoFacetList = $this->getFacetValues(2);
    $this->tpl->assign('htoFacetList', $htoFacetList);
    $domFacetList = $this->getFacetValues(5);
    $this->tpl->assign('domFacetList', $domFacetList);
  }


  function getDataDiff($thisDataArray, $lastDataArray) {
    if ((is_array($thisDataArray) && count($thisDataArray)>0)) {
    foreach ($thisDataArray as $key => $value) {
      if (is_array($value)) {
        $diffDataArray[$key] = $this->getDataDiff($thisDataArray[$key], $lastDataArray[$key]);
      } else {
        if ($thisDataArray[$key]>0) {
          $diffDataArray[$key] = number_format (100.0 * (($thisDataArray[$key]) - ($lastDataArray[$key])) / ($lastDataArray[$key]), 2);
        } else {
          $diffDataArray[$key] = 0;
        }
      }
    }
    }
    return $diffDataArray;
  }


  /**
  * output HTML version of the report
  *
  */
  function outputHtmlReport($agg_type, $loc_type, $loc_id, $time_period) {
    $mod_agg_type = $agg_type;
    if ($mod_agg_type==null || $mod_agg_type=="") {
        $mod_agg_type=$this->getInferredAggType($time_period);
    }
    $this->createReport($mod_agg_type, $loc_type, $loc_id, $time_period);
    $this->tpl->display($this->getTemplate($mod_agg_type));
  }

  /**
  * infer agg_type based on time_period (YYYY or YYYYMM format)
  *
  */
  function getInferredAggType($time_period) {
    if (strlen($time_period)==6) {
      $agg_type = "MONTHLY";
    } else if (strlen($time_period)==4) {
      $agg_type = "YEARLY";
    } else {
      throw new Exception("Invalid Time Period");
    }
    return $agg_type;
  }

  /**
  * get report template based on time_period (YYYY or YYYYMM format)
  *
  */
  function getTemplate($agg_type) {
    $template = strtolower("hot_fusion_$agg_type.tpl");
    return $template;
  }

  /**
  * get previous year for time_period (YYYY or YYYYMM format)
  *
  */
  function getPreviousYearTimePeriod($time_period) {
    if (strlen($time_period)==6) {
      $prev_time_period = $time_period - 100; // subtract 1 year and 00 months
    } else if (strlen($time_period)==4) {
      $prev_time_period = $time_period - 1; // subtract 1 year 
    } else {
      throw new Exception("Invalid Time Period");
    }
    return $prev_time_period;
  }


  /**
  * generate PDF version of the report
  *
  * @param array $formvars the form variables
  */
  function outputPdfReport($agg_type, $loc_type, $loc_id, $time_period, $filename, $title) {

    // assign error message
    $mod_agg_type = $agg_type;
    if ($mod_agg_type==null || $mod_agg_type=="") {
        $mod_agg_type=$this->getInferredAggType($time_period);
    }
    $this->createReport($mod_agg_type, $loc_type, $loc_id, $time_period);
    $html = $this->tpl->fetch($this->getTemplate($mod_agg_type));
    
    try {
        $wkhtmltopdf = new Wkhtmltopdf(array('path' => RBI_REPORT_DIR . '/public/'));
        $wkhtmltopdf->setTitle($title);
        $wkhtmltopdf->setHtml($html);
        $wkhtmltopdf->setPageSize(Wkhtmltopdf::SIZE_LETTER);
        #$wkhtmltopdf->output(Wkhtmltopdf::MODE_SAVE, $filename);
        $wkhtmltopdf->output(Wkhtmltopdf::MODE_EMBEDDED, $filename);
    } catch (Exception $e) {
        echo $e->getMessage();
    }
  }


  function getAggTable($agg_type, $threedim=false) {
      $table_name = "facet_aggregations";
      if ($agg_type=="QTD" || $agg_type=="YTD" || $agg_type=="QUARTERLY") {
        $table_name = $table_name . "_" . $agg_type;
      }
      if ($threedim) {
        $table_name = $table_name . "_3fd";
      }
      $table_name = strtolower($table_name);
      return $table_name;
  }

  /**
  * get the core aggregation data
  */
  function getCoreAggregationData($agg_type, $loc_type, $loc_id, $time_period, $ft_id=1, $f_id=null) {
    try {
      $table_name = $this->getAggTable($agg_type);

      $time_col = "time_period";
      #if ($table_name!="facet_aggregations") {
      #   $time_col="month_id";
      #}

      if ($f_id==null) {
        $query="select * from $table_name where loc_type=$loc_type and loc_id=$loc_id and $time_col=$time_period and f_id in (select f_id from facet_values where ft_id=$ft_id)";
      } else {
        $query="select * from $table_name where loc_type=$loc_type and loc_id=$loc_id and $time_col=$time_period and f_id=$f_id";
      }
//echo $query . "<BR>\n";

      foreach($this->pdo->query($query) as $row) {
        if ($f_id==null && $ft_id!=1) {
         $rows['fid' . $row['f_id']] = $row;
        } else {
         $rows[] = $row;
        }
      }
    } catch (PDOException $e) {
      print "Error!: " . $e->getMessage();
      return false;
    }   
    if ($f_id==null && $ft_id!=1) {
      return $rows;
    } else {
      return $rows[0];   
    }
  }

  function getDetailAggregationData($agg_type, $loc_type, $loc_id, $time_period, $ft_id=1, $ft2_id, $ft3_id) {
    try {
      $table_name = $this->getAggTable($agg_type,true);

      $time_col = "time_period";
      #if ($table_name!="facet_aggregations_3fd") {
      #   $time_col="month_id";
      #}

      $query="select * from $table_name where loc_type=$loc_type and loc_id=$loc_id and $time_col=$time_period and f_id in (select f_id from facet_values where ft_id=$ft_id) and f_id2 in (select f_id from facet_values where ft_id=$ft2_id) and f_id3 in (select f_id from facet_values where ft_id=$ft3_id)";
//echo $query . "<BR>\n";
      foreach($this->pdo->query($query) as $row) {
        $rows['fid' . $row['f_id']]['fid' . $row['f_id2']]['fid' . $row['f_id3']] = $row;
      }
      if (count($rows)==0) {
        #throw new Exception("No Rows Found -- missing data in source for query \"$query\"");
        $rows[] = array();
      }
    } catch (PDOException $e) {
      print "Error!: " . $e->getMessage();
      return false;
    }
    return $rows;
  }

  function getLocationData($loc_type, $loc_id) {
    try {
      if ($loc_type==5) {
        $row[l_id] = $loc_id;
        $row[lt_id] = $loc_type;
        $row[l_desc] = $loc_id;
        $rows[0] = $row;
      } else {
        $query="select * from location_values where lt_id=$loc_type and l_id=$loc_id";
        foreach($this->pdo->query($query) as $row) {
          $rows[] = $row;
        }
        if (count($rows)==0) {
          throw new Exception("No Rows Found -- missing data in source.");
        }
      }
    } catch (PDOException $e) {
      print "Error!: " . $e->getMessage();
      return false;
    }
    return $rows[0];
  }

  function getFacetValues($ft_id) {
    try {
      $query="select * from facet_values where ft_id=$ft_id";
      foreach($this->pdo->query($query) as $row) {
        $rows[] = $row;
      }
      if (count($rows)==0) {
        throw new Exception("No Rows Found -- missing data in source.");
      }
    } catch (PDOException $e) {
      print "Error!: " . $e->getMessage();
      return false;
    }
    return $rows;
  }

  function getLocationTypes() {
    try {
      $query="select * from location_types";
      foreach($this->pdo->query($query) as $row) {
        $rows[] = $row;
      }
      if (count($rows)==0) {
        throw new Exception("No Rows Found -- missing data in source.");
      }
    } catch (PDOException $e) {
      print "Error!: " . $e->getMessage();
      return false;
    }
    return $rows;
  }

  function getLocationValues($lt_id) {
    try {
      $query="select * from location_values where lt_id=$lt_id order by l_id";
      if ($lt_id==4) {
        $query="select * from location_values where lt_id=$lt_id order by substr(l_desc,instr(l_desc, ',')+2), l_desc";
      }
      if ($lt_id==5) {
        $query="select distinct zip as l_id, zip as l_desc from ZIP_MASTER_LIST order by zip";
      }
      foreach($this->pdo->query($query) as $row) {
        $rows[] = $row;
      }
      if (count($rows)==0) {
        throw new Exception("No Rows Found -- missing data in source.");
      }
    } catch (PDOException $e) {
      print "Error!: " . $e->getMessage();
      return false;
    }
    return $rows;
  }

  function getReportMonths() {
    try {
      $query="select distinct time_period from facet_aggregations where time_period>2500  order by time_period desc";
      foreach($this->pdo->query($query) as $row) {
        $rows[] = $row;
      }
      if (count($rows)==0) {
        throw new Exception("No Rows Found -- missing data in source.");
      }
    } catch (PDOException $e) {
      print "Error!: " . $e->getMessage();
      return false;
    }
    return $rows;
  }

  function getReportYears() {
    try {
      $query="select distinct time_period from facet_aggregations where time_period<2500  order by time_period desc";
      foreach($this->pdo->query($query) as $row) {
        $rows[] = $row;
      }
      #if (count($rows)==0) {
      #  throw new Exception("No Rows Found -- missing data in source.");
      #}
    } catch (PDOException $e) {
      print "Error!: " . $e->getMessage();
      return false;
    }
    return $rows;
  }

}

?>
